% Script to reproduce the experiment of Fig. 2 of the paper

% Define dimension of the matrix
n = 2000;
pmax = n-1;
repeats = 10;
x = linspace(0,1,n);
h = x(2)-x(1);

% Matrix of the discretized Laplacian Green's function
L = (1/h^2)*gallery('tridiag',n,-1,2,-1);

% Define matrix A
K = 10;
a = @(x)sin(5*pi*x);
A = L - K^2*spdiags(a(x)',0,n,n);
I = eye(n,n);

% Compute SVD of A
[~,S,~] = svd(full(A));
S = diag(S);
S = flipud(S.^(-1));
NormA = sqrt(sum(S.^2));

% Compute the SVD of the covariance matrix L
V_K = sqrt(2)*sin(x'.*(1:n)*pi);
Factor = sqrt(h)*(1:n).^(-1)/pi;
Cov = V_K*diag(Factor);

% Define arrays
Error_svd = zeros(pmax, repeats);
Error_prior_kernel = zeros(pmax, repeats);
Error_best = zeros(pmax);
timing_svd = zeros(pmax,repeats);
timing_prior_kernel = zeros(pmax,repeats);

for i = 1:repeats
    sprintf("##### Repeat = %d / %d #####",i,repeats)
    for p = pmax:pmax
        sprintf("p = %d / %d",p,pmax)

        % Use standard Gaussian matrix
        tic;
        X = randn(n,p);
        Y = A \ X;
        [Q,~] = qr(Y,0);
        B = norm((I-Q*Q') / A,'fro');
        timing_svd(p,i) = toc;
        Error_svd(p,i) = B / NormA;

        % Use covariance matrix L to generate random inputs
        tic;
        X_rand = Cov*randn(n,p);
        Y = A \ X_rand;
        [Q,~] = qr(Y,0);
        B = norm((I-Q*Q') / A,'fro');
        timing_prior_kernel(p,i) = toc;
        Error_prior_kernel(p,i) = B / NormA;

        % Best approximation error
        Sk = sqrt(sum(S(p+1:end).^2));
        Error_best(p) = Sk / NormA;
    end
end

save("exp1.mat", "Error_svd", "Error_prior_kernel", "Error_best","timing_svd","timing_prior_kernel")
